<?php
if (!defined('ABSPATH')) {
    exit;
}

// ============================================================
// 1. وظائف حماية IP وحدود الطلب
// ============================================================

// تسجيل الطلب لعنوان IP معين
function amcod_log_order_for_ip($ip) {
    $order_limits = get_option('amcod_order_limits', array());
    $current_time = time();
    
    // الحصول على مدة الحد من الإعدادات
    $limit_duration = get_option('amcod_order_limit_time', 24) * 60 * 60;

    if (isset($order_limits[$ip])) {
        $order_limits[$ip] = array_filter($order_limits[$ip], function($timestamp) use ($current_time, $limit_duration) {
            return ($current_time - $timestamp) <= $limit_duration;
        });
    }

    $order_limits[$ip][] = $current_time;
    update_option('amcod_order_limits', $order_limits);
}

// التحقق مما إذا كان IP قد تجاوز الحد
function amcod_has_exceeded_order_limit($ip) {
    $order_limits = get_option('amcod_order_limits', array());
    $current_time = time();
    
    $limit_duration = get_option('amcod_order_limit_time', 24) * 60 * 60;
    $order_limit_count = get_option('amcod_order_limit_count', 0);

    if ($order_limit_count <= 0) {
        return false; 
    }

    if (isset($order_limits[$ip])) {
        $order_limits[$ip] = array_filter($order_limits[$ip], function($timestamp) use ($current_time, $limit_duration) {
            return ($current_time - $timestamp) <= $limit_duration;
        });

        if (count($order_limits[$ip]) >= $order_limit_count) {
            amcod_block_ip_for_time($ip, $limit_duration);
            return true;
        }
    }

    return false;
}

// حظر IP لمدة معينة
function amcod_block_ip_for_time($ip, $block_duration = 86400) {
    $blocked_ips = get_option('amcod_blocked_ips', array());
    $blocked_ips[$ip] = time() + $block_duration;
    update_option('amcod_blocked_ips', $blocked_ips);
}

// التحقق من حالة الحظر
function amcod_check_ip_order_limits($customer_ip) {
    $blocked_ips = get_option('amcod_blocked_ips', array());

    if (isset($blocked_ips[$customer_ip])) {
        $current_time = time();
        if ($blocked_ips[$customer_ip] == 0 || $blocked_ips[$customer_ip] > $current_time) {
            return array('status' => 'blocked', 'message' => __('لقد تم حظرك مؤقتاً بسبب تجاوز حد الطلبات.', 'amcod'));
        } else {
            unset($blocked_ips[$customer_ip]);
            update_option('amcod_blocked_ips', $blocked_ips);
        }
    }
    
    if (get_option('amcod_order_limit_count', 0) <= 0) {
        delete_option('amcod_blocked_ips');
    }

    return array('status' => 'ok');
}

// ============================================================
// 2. معالجة تقديم النموذج
// ============================================================

function amcod_handle_custom_checkout_form_submission() {
    if ($_SERVER['REQUEST_METHOD'] === 'POST' 
        && isset($_POST['quantity']) 
        && isset($_POST['product_id'])) {

        $customer_ip = $_SERVER['REMOTE_ADDR'];

        // التحقق من الحظر
        $ip_check = amcod_check_ip_order_limits($customer_ip);
        if ($ip_check['status'] !== 'ok') {
            wc_add_notice($ip_check['message'], 'error');
            return;
        }

        // التحقق من تجاوز الحدود
        if (amcod_has_exceeded_order_limit($customer_ip)) {
            wc_add_notice(__('لقد وصلت إلى الحد الأقصى للطلب. يرجى المحاولة لاحقاً.', 'amcod'), 'error');
            return;
        }

        // جلب البيانات وتنظيفها
        $first_name = get_option('amcod_disable_first_name') === 'yes' ? '' : (isset($_POST['billing_first_name']) ? sanitize_text_field($_POST['billing_first_name']) : '');
        $phone      = get_option('amcod_disable_phone') === 'yes' ? '' : (isset($_POST['billing_phone']) ? sanitize_text_field($_POST['billing_phone']) : '');
        $email      = get_option('amcod_disable_email') === 'yes' ? '' : (isset($_POST['billing_email']) ? sanitize_email($_POST['billing_email']) : '');
        $address_1  = get_option('amcod_disable_address') === 'yes' ? '' : (isset($_POST['billing_address_1']) ? sanitize_text_field($_POST['billing_address_1']) : '');
        $state      = get_option('amcod_disable_state') === 'yes' ? '' : (isset($_POST['billing_state']) ? sanitize_text_field($_POST['billing_state']) : '');
        $city       = get_option('amcod_disable_city') === 'yes' ? '' : (isset($_POST['billing_city']) ? sanitize_text_field($_POST['billing_city']) : '');
        
        $quantity     = intval($_POST['quantity']);
        $product_id   = intval($_POST['product_id']);
        $variation_id = isset($_POST['variation_id']) ? intval($_POST['variation_id']) : 0;
        $enable_security_check = get_option('amcod_enable_security_check') === 'yes';

        // التحقق من المنتج والمخزون
        $product = wc_get_product($product_id);
        if (!$product || !$product->is_in_stock()) {
            wc_add_notice(get_option('amcod_error_product_out_of_stock', 'عذراً، المنتج غير متوفر.'), 'error');
            return;
        }

        if ($variation_id) {
            $variation = wc_get_product($variation_id);
            if ($variation && !$variation->is_in_stock()) {
                wc_add_notice(get_option('amcod_error_variation_out_of_stock', 'عذراً، هذا النوع غير متوفر.'), 'error');
                return;
            }
        }

        // تجميع السمات (Attributes)
        $attributes = array();
        foreach ($_POST as $key => $value) {
            if (strpos($key, 'attribute_') === 0) {
                $attributes[$key] = sanitize_text_field($value);
            }
        }

        // التحقق من كود الأمان
        if ($enable_security_check) {
            if (!isset($_POST['security_code']) || !isset($_SESSION['security_code']) || $_POST['security_code'] !== $_SESSION['security_code']) {
                wc_add_notice(__('كود الأمان غير صحيح.', 'amcod'), 'error');
                return;
            }
        }

        // التحقق من الهاتف
        if (!empty($phone)) {
            if (!preg_match('/^\d{10}$/', $phone)) {
                wc_add_notice(get_option('amcod_error_invalid_phone', 'رقم الهاتف غير صالح.'), 'error');
                return;
            }
        }

        // التحقق من الحقول المطلوبة
        if (
            (get_option('amcod_disable_first_name') !== 'yes' && empty($first_name)) ||
            (get_option('amcod_disable_state') !== 'yes' && empty($state)) ||
            (get_option('amcod_disable_phone') !== 'yes' && empty($phone)) ||
            empty($quantity) ||
            empty($product_id)
        ) {
            wc_add_notice(get_option('amcod_error_required_fields', 'الرجاء ملء جميع الحقول.'), 'error');
            return;
        }

        // التحقق من السمات
        foreach ($attributes as $key => $value) {
            if (empty($value)) {
                wc_add_notice(get_option('amcod_error_select_attributes', 'الرجاء اختيار المواصفات.'), 'error');
                return;
            }
        }

        // إنشاء الطلب
        $address = array(
            'first_name' => $first_name,
            'email'      => $email,
            'phone'      => $phone,
            'address_1'  => $address_1,
            'state'      => $state,
            'city'       => $city,
            'country'    => WC()->countries->get_base_country(),
        );

        $order = wc_create_order();
        $order->set_address($address, 'billing');
        $order->set_address($address, 'shipping'); // إضافة الشحن للعنوان أيضاً

        // إضافة المنتج للطلب
        $variation_obj = $variation_id ? wc_get_product($variation_id) : null;
        if ($variation_obj) {
            $order->add_product($variation_obj, $quantity);
        } else {
            $order->add_product($product, $quantity);
        }

        // إضافة الملاحظات
        if (isset($_POST['order_note']) && !empty($_POST['order_note'])) {
            $order->set_customer_note(sanitize_textarea_field($_POST['order_note']));
        }

        // --- حساب الشحن وإضافته للطلب ---
        // إعداد بيئة الشحن
        WC()->customer->set_shipping_location(WC()->countries->get_base_country(), $state, '', $city);
        WC()->customer->set_shipping_address_1($address_1);
        
        // حساب التكلفة
        WC()->cart->empty_cart(); // تفريغ السلة الوهمية
        // إضافة المنتج للسلة لحساب الشحن بدقة
        if ($variation_id) {
             WC()->cart->add_to_cart($product_id, $quantity, $variation_id, $attributes);
        } else {
             WC()->cart->add_to_cart($product_id, $quantity);
        }

        WC()->shipping()->reset_shipping();
        WC()->shipping()->calculate_shipping(WC()->cart->get_shipping_packages());
        $shipping_packages = WC()->shipping->get_packages();

        // إضافة طريقة الشحن المختارة
        if (!empty($_POST['shipping_method'])) {
            $chosen_method_id = sanitize_text_field($_POST['shipping_method']);
            
            // البحث عن الطريقة وتفاصيلها
            if (!empty($shipping_packages) && isset($shipping_packages[0]['rates'][$chosen_method_id])) {
                $rate = $shipping_packages[0]['rates'][$chosen_method_id];
                
                $shipping_item = new WC_Order_Item_Shipping();
                $shipping_item->set_method_title($rate->label);
                $shipping_item->set_method_id($chosen_method_id);
                $shipping_item->set_total($rate->cost);
                $order->add_item($shipping_item);
            }
        }
        // --------------------------------

        $order->calculate_totals();
        $order->set_payment_method('cod');
        $order->set_payment_method_title('الدفع عند الاستلام');
        $order->save();

        // حفظ السمات كبيانات وصفية
        foreach ($order->get_items() as $item) {
            foreach ($attributes as $key => $value) {
                $clean_key = str_replace('attribute_', '', $key); // تنظيف الاسم
                $item->update_meta_data($clean_key, $value);
            }
            $item->save();
        }

        $order->update_status('processing', 'AM COD Order');
        
        // تسجيل الطلب للـ IP
        amcod_log_order_for_ip($customer_ip);
        
        WC()->cart->empty_cart();
        wp_redirect($order->get_checkout_order_received_url());
        exit;
    }
}
add_action('template_redirect', 'amcod_handle_custom_checkout_form_submission');

// تصفية بوابات الدفع (إجبار الدفع عند الاستلام إذا لم يوجد غيره)
add_filter('woocommerce_available_payment_gateways', function ($gateways) {
    if (empty($gateways) && isset(WC()->payment_gateways()->payment_gateways()['cod'])) {
        $gateways['cod'] = WC()->payment_gateways()->payment_gateways()['cod'];
    }
    return $gateways;
});

// التأكد من أن السلة تحتاج دفع (لتجنب الأخطاء)
add_filter('woocommerce_cart_needs_payment', function ($needs_payment) {
    $gateways = WC()->payment_gateways()->get_available_payment_gateways();
    return empty($gateways) ? false : $needs_payment;
}, 10, 2);
?>